/*
 * Copyright 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package calendrier.maree.presentation


import android.Manifest
import android.annotation.SuppressLint
import android.app.AlertDialog
import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.pm.PackageManager
import android.os.Build
import android.os.Bundle
import android.util.Log
import android.view.View
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.annotation.RequiresApi
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Anchor
import androidx.compose.material.icons.filled.DateRange
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.vectorResource
import androidx.core.app.ActivityCompat
import androidx.core.splashscreen.SplashScreen.Companion.installSplashScreen
import androidx.navigation.NavHostController
import androidx.wear.compose.material.ButtonDefaults
import androidx.wear.compose.material.ChipDefaults.chipColors
import androidx.wear.compose.material.Icon
import androidx.wear.compose.material.MaterialTheme
import androidx.wear.compose.material.Text
import androidx.wear.compose.material.dialog.Dialog
import androidx.wear.compose.navigation.SwipeDismissableNavHost
import androidx.wear.compose.navigation.composable
import androidx.wear.compose.navigation.rememberSwipeDismissableNavController
import calendrier.maree.Network.DownloadCallback
import calendrier.maree.Network.PlayActiveScenario
import calendrier.maree.data.MareeTable
import calendrier.maree.presentation.components.CustomTimeText
import calendrier.maree.presentation.navigation.Screen
import calendrier.maree.presentation.theme.WearAppTheme
import calendrier.maree.presentation.ui.Clock
import calendrier.maree.presentation.ui.watchlist.WatchListScreen1
import calendrier.maree.presentation.ui.watchlist.WatchListScreenDt
import calendrier.maree.R
import com.google.android.gms.location.FusedLocationProviderClient
import com.google.android.gms.location.LocationServices
import com.google.android.gms.location.Priority
import com.google.android.horologist.composables.DatePicker
import com.google.android.horologist.composables.TimePicker
import com.google.android.horologist.compose.layout.AppScaffold
import com.google.android.horologist.compose.layout.ScalingLazyColumn
import com.google.android.horologist.compose.layout.ScalingLazyColumnDefaults
import com.google.android.horologist.compose.layout.ScalingLazyColumnState
import com.google.android.horologist.compose.layout.ScreenScaffold
import com.google.android.horologist.compose.layout.rememberResponsiveColumnState
import com.google.android.horologist.compose.material.AlertContent
import com.google.android.horologist.compose.material.Button
import com.google.android.horologist.compose.material.Chip
import com.google.android.horologist.compose.material.ListHeaderDefaults.firstItemPadding
import com.google.android.horologist.compose.material.ResponsiveListHeader
import java.time.LocalDateTime


/**
 * Compose for Wear OS app that demonstrates how to use Wear specific Scaffold, Navigation,
 * curved text, Chips, and many other composables.
 *
 * Displays different text at the bottom of the landing screen depending on shape of the device
 * (round vs. square/rectangular).
 */
class MainActivity : ComponentActivity(), DownloadCallback {
    //private lateinit var jankPrinter: JankPrinter
    internal lateinit var navController: NavHostController
    private lateinit var fusedLocationClient: FusedLocationProviderClient
    private val LOCATION_PERMISSION_REQUEST = 100
    lateinit var mCallback: DownloadCallback
    var mMareeTable: MareeTable = MareeTable()
    var mMaree: String = ""
    private var clock: Clock? = null


    override fun onCreate(savedInstanceState: Bundle?) {
        installSplashScreen()
        super.onCreate(savedInstanceState)
        // Initialisation FusedLocationProvider
        fusedLocationClient = LocationServices.getFusedLocationProviderClient(this)

        mCallback = this
        val sharedPref = getSharedPreferences("app_prefs", MODE_PRIVATE)
        val savedPortId = sharedPref.getString("last_port_id", "LA_ROCHELLE-PALLICE") // Valeur par défaut
        if (savedPortId != null) {
            mMareeTable.SetPortName(savedPortId)
        }
        PlayActiveScenario(
            mMareeTable.GetShomUrl(),
            mCallback
        )
        setTheme(android.R.style.Theme_DeviceDefault)

        setContentView(R.layout.sample_main)
        clock = findViewById<View>(R.id.clock) as Clock
        clock!!.setOnClickListener {
            clock!!.onPause()
            runMenu()
        }

    }

    @RequiresApi(Build.VERSION_CODES.S)
    override fun UpdateReponse(vRet: String) {
        mMaree = vRet
        mMareeTable.LoadMareeTable(mMaree)
        mMareeTable.LoadDatesMaree()
        mMareeTable.SetIndexMareeDate(0)
        if (clock != null){
            clock!!.SetMareeTable(mMareeTable)
        }
        checkLocationPermission()
    }

    @RequiresApi(Build.VERSION_CODES.S)
    override fun checkLocationPermission() {
        if (ActivityCompat.checkSelfPermission(this, Manifest.permission.ACCESS_FINE_LOCATION)
            != PackageManager.PERMISSION_GRANTED
        ) {
            ActivityCompat.requestPermissions(
                this,
                arrayOf(Manifest.permission.ACCESS_FINE_LOCATION),
                LOCATION_PERMISSION_REQUEST
            )
        } else {
            // getSingleLocation()
        }
    }

    @SuppressLint("MissingPermission")
    fun getSingleLocation() {
        if (ActivityCompat.checkSelfPermission(this, Manifest.permission.ACCESS_FINE_LOCATION)
            != PackageManager.PERMISSION_GRANTED) {
            ActivityCompat.requestPermissions(
                this,
                arrayOf(Manifest.permission.ACCESS_FINE_LOCATION),
                100
            )
            return
        }

        fusedLocationClient.getCurrentLocation(Priority.PRIORITY_HIGH_ACCURACY, null)
            .addOnSuccessListener { location ->
                location?.let {
                    val lat = it.latitude
                    val lon = it.longitude
                    Log.d("LOCATION", "Latitude: $lat, Longitude: $lon")

                    mMareeTable.GetIndexMyPort(lat, lon)
                    showMessage(this, "Port Name", "Port le plus proche : " +mMareeTable.GetPortName())
                    // Log.d("PORT", "Port le plus proche : index $closestPortIndex")
                } ?: Log.d("LOCATION", "Location is null")
            }
            .addOnFailureListener {
                Log.e("LOCATION", "Failed to get location", it)
            }
    }



    private fun showMessage(context: Context, title: String, message: String) {
        AlertDialog.Builder(context)
            .setTitle(title)
            .setMessage(message)
            .setPositiveButton("OK", null)
            .show()
    }

    override fun onRequestPermissionsResult(
        requestCode: Int, permissions: Array<out String>, grantResults: IntArray
    ) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        if (requestCode == LOCATION_PERMISSION_REQUEST &&
            grantResults.isNotEmpty() &&
            grantResults[0] == PackageManager.PERMISSION_GRANTED
        ) {
            getSingleLocation()
        }
    }

    override fun onResume() {
        super.onResume()
        clock!!.onResume()

    }

    override fun onPause() {
        super.onPause()
        clock!!.onPause()
    }


    fun runMenu(){
        //Toast.makeText(this,"Button Pressed",Toast.LENGTH_SHORT).show()
        setContent {
            navController = rememberSwipeDismissableNavController()
            clock?.let {
                WearApp(
                    swipeDismissableNavController = navController,
                    vMareeTable = mMareeTable,
                    mCallback = mCallback,
                    clock  = it
                )
            }
        }
    }
    override fun loadClock(){
        setContentView(R.layout.sample_main)
        clock = findViewById<View>(R.id.clock) as Clock
        clock!!.onResume()
        clock!!.SetMareeTable(mMareeTable)
        clock!!.setOnClickListener {
            clock!!.onPause()
            runMenu()
        }
    }
}




@Composable
fun WearApp(
    modifier: Modifier = Modifier,
    swipeDismissableNavController: NavHostController = rememberSwipeDismissableNavController(),
    vMareeTable : MareeTable,
    mCallback: DownloadCallback,
    clock : Clock
) {
    val navController = rememberSwipeDismissableNavController()
    var dateTimeForUserInput by remember { mutableStateOf(LocalDateTime.now().plusHours(2) ) }
    val context = LocalContext.current
    WearAppTheme {
        AppScaffold (modifier=Modifier, timeText =
        {   CustomTimeText(
            visible = false
            )
        }
        ){
            SwipeDismissableNavHost(navController = navController, startDestination = "menu") {
                composable("menu") {
                    ListScreen(navController,vMareeTable,mCallback)
                }
                composable("list") {
                    ListScreen(navController,vMareeTable,mCallback)
                }
                composable(Screen.Time24hPicker.route) {
                    TimePicker(
                        onTimeConfirm = {
                            swipeDismissableNavController.popBackStack()
                            dateTimeForUserInput = it.atDate(dateTimeForUserInput.toLocalDate())
                            vMareeTable.vCurrentTimeToSet = dateTimeForUserInput.toLocalTime()
                            navController.navigate("menu")},
                        time = vMareeTable.vCurrentTimeToSet //dateTimeForUserInput.toLocalTime()

                    )
                }
                composable(Screen.DatePicker.route) {
                DatePicker(
                    // lecture de la date selectionnée
                    onDateConfirm = {
                        swipeDismissableNavController.popBackStack()
                        //swipeDismissableNavController.navigate("menu")
                        dateTimeForUserInput = it.atTime(dateTimeForUserInput.toLocalTime())
                        vMareeTable.vCurrentDateMaree = dateTimeForUserInput.toLocalDate()

                        PlayActiveScenario(
                            vMareeTable.GetShomUrl(),
                            mCallback
                        )
                        //navController.navigate("menu")
                        mCallback.loadClock()
                    },
                )
                }

                composable(
                    route = Screen.WatchListDt.route
                ) {
                    val vignetteVisible = rememberSaveable { mutableStateOf(true) }

                    WatchListScreenDt(
                        columnState = ScalingLazyColumnState(),
                        showVignette = vignetteVisible.value,
                        onClickVignetteToggle = { showVignette ->
                            vignetteVisible.value = showVignette
                        },
                        onClickWatch = { id ->
                            swipeDismissableNavController.popBackStack()
                            vMareeTable.SetIndexMareeDate(id)
                            vMareeTable.vCurrentTimeToSet = LocalDateTime.now().toLocalTime().plusHours(2)
                            PlayActiveScenario(
                                vMareeTable.GetShomUrl(),
                                mCallback
                            )
                            //navController.navigate("menu")
                            mCallback.loadClock()
                        }, vMareeTable
                    )
                }

                composable(
                    route = Screen.WatchList.route
                ) {
                    val vignetteVisible = rememberSaveable { mutableStateOf(true) }

                       WatchListScreen1(
                        columnState = ScalingLazyColumnState(),
                        showVignette = vignetteVisible.value,
                        onClickVignetteToggle = { showVignette ->
                            vignetteVisible.value = showVignette
                        },
                        onClickWatch = { id ->
                            swipeDismissableNavController.popBackStack()
                            vMareeTable.SetIndexPortName(id)
                            val sharedPref = context.getSharedPreferences("app_prefs", MODE_PRIVATE)
                            with(sharedPref.edit()) {
                                putString("last_port_id", vMareeTable.GetPortId(id))
                                apply()
                            }
                            PlayActiveScenario(
                                vMareeTable.GetShomUrl(),
                                mCallback
                            )
                            //navController.navigate("menu")
                            mCallback.loadClock()
                        }, vMareeTable
                    )
                }
            }
        }
    }
}

@SuppressLint("ResourceType")
@Composable
fun ListScreen(navController : NavHostController, vTableMareeTable: MareeTable, mCallback: DownloadCallback,) {
    var showDialog by remember { mutableStateOf(false) }

    /*
     * Specifying the types of items that appear at the start and end of the list ensures that the
     * appropriate padding is used.
     */
    val columnState = rememberResponsiveColumnState(
        contentPadding = ScalingLazyColumnDefaults.padding(
            first = ScalingLazyColumnDefaults.ItemType.Text,
            last = ScalingLazyColumnDefaults.ItemType.SingleButton
        )
    )
    val context = LocalContext.current
    val activity = context as MainActivity

    ScreenScaffold(scrollState = columnState) {
        /*
         * The Horologist [ScalingLazyColumn] takes care of the horizontal and vertical
         * padding for the list, so there is no need to specify it, as in the [GreetingScreen]
         * composable.
         */
        ScalingLazyColumn(
            columnState = columnState
        ) {
            item {
                Text(text = "PARAMETRES")
            }
            item {
                Chip(
                    modifier = Modifier,
                    colors = chipColors(backgroundColor = MaterialTheme.colors.secondaryVariant),
                    icon = {
                        Icon(
                            imageVector = Icons.Default.DateRange,
                            contentDescription = "search",
                            modifier = Modifier
                        )
                    },
                    label = "Dates des Marées", onClick = { navController.navigate(Screen.WatchListDt.route)})
            }
            item {
                Chip(
                    modifier = Modifier,
                    colors = chipColors(backgroundColor = MaterialTheme.colors.secondaryVariant),
                    icon = {
                        Icon(
                            imageVector = Icons.Default.Anchor,
                            contentDescription = "search",
                            modifier = Modifier
                        )
                    },
                    label = "Changer de Port", onClick = { navController.navigate(Screen.WatchList.route)})
            }
            item {
                Chip(
                    modifier = Modifier,
                    colors = chipColors(backgroundColor = MaterialTheme.colors.secondaryVariant),
                    icon = {
                        Icon(
                            imageVector = ImageVector.vectorResource(id = R.drawable.baseline_gps_fixed_24),
                            contentDescription = "search",
                            modifier = Modifier
                        )
                    },
                    label = "Trouver son Port", onClick = {
                        activity.getSingleLocation()


                    })
            }
            item {
                ResponsiveListHeader(contentPadding = firstItemPadding()) {
                    Button(
                        imageVector = ImageVector.vectorResource(id = R.drawable.baseline_close_24),
                        contentDescription = "Retour",
                        onClick = {

                            PlayActiveScenario(
                                vTableMareeTable.GetShomUrl(),
                                mCallback
                            )
                            vTableMareeTable.vCurrentTimeToSet = LocalDateTime.now().toLocalTime().plusHours(2)
                            mCallback.loadClock()
                        },

                        colors = ButtonDefaults.secondaryButtonColors()
                    )
                }
            }
        }
    }

    SampleDialog(
        showDialog = showDialog,
        onDismiss = { showDialog = false },
        onCancel = {},
        onOk = {}
    )
}


@Composable
fun SampleDialog(
    showDialog: Boolean,
    onDismiss: () -> Unit,
    onCancel: () -> Unit,
    onOk: () -> Unit
) {
    val state = rememberResponsiveColumnState()

    Dialog(
        showDialog = showDialog,
        onDismissRequest = onDismiss,
        scrollState = state.state
    ) {
        SampleDialogContent(onCancel, onDismiss, onOk)
    }
}
@Composable
fun SampleDialogContent(
    onCancel: () -> Unit,
    onDismiss: () -> Unit,
    onOk: () -> Unit
) {
    AlertContent(
        icon = {},
        title = "Title",
        onCancel = {
            onCancel()
            onDismiss()
        },
        onOk = {
            onOk()
            onDismiss()
        }
    ) {
        item {
            Text(text = "An unknown error occurred during the request.")
        }
    }
}


