/*
 * Copyright 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package calendrier.maree.presentation.ui.watchlist

import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.foundation.rememberScrollState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.wear.compose.foundation.lazy.ScalingLazyColumn
import androidx.wear.compose.foundation.lazy.items
import androidx.wear.compose.material.ChipDefaults
import androidx.wear.compose.material.MaterialTheme
import androidx.wear.compose.material.Text
import calendrier.maree.data.MareeTable
import calendrier.maree.data.PortsMaree
import calendrier.maree.data.WatchLocalDataSource
import calendrier.maree.data.WatchModelDt
import calendrier.maree.presentation.components.WatchAppCards
import calendrier.maree.presentation.components.WatchAppChip
import calendrier.maree.presentation.util.formatDateMaree
import com.google.android.horologist.compose.layout.ScalingLazyColumn
import com.google.android.horologist.compose.layout.ScalingLazyColumnState
import com.google.android.horologist.compose.layout.ScreenScaffold
import com.google.android.horologist.compose.material.ToggleChip
import kotlinx.coroutines.launch

/**
 * Displays a list of watches plus a [ToggleChip] at the top to display/hide the Vignette around
 * the screen. The list is powered using a [ScalingLazyColumn].
 */
@Composable
fun WatchListScreen(
    watches: List<PortsMaree>,
    columnState: ScalingLazyColumnState,
    showVignette: Boolean,
    onClickVignetteToggle: (Boolean) -> Unit,
    onClickWatch: (Int) -> Unit,
    modifier: Modifier = Modifier,
    //vIndexScroll: Int = 0
    vMareeTable : MareeTable

) {

    val state = rememberScrollState()
    val listState = rememberLazyListState()
    val coroutineScope = rememberCoroutineScope()
    val vIndexScroll = vMareeTable.currentIndexPortName()
    ScreenScaffold(scrollState = columnState) {
        /*
         * The Horologist [ScalingLazyColumn] takes care of the horizontal and vertical
         * padding for the list, so there is no need to specify it, as in the [GreetingScreen]
         * composable.
         */
        //LaunchedEffect(Unit) { state.animateScrollTo(100) }
        ScalingLazyColumn(
            modifier = modifier.fillMaxSize(),
            columnState = columnState,
        ) {
            item {
                Text(text = "Liste des Ports")
            }
            // Displays all watches.
            //items(watches,key={it.modelId}) { watch ->
            items(watches) { watch ->


                var vColorPort =ChipDefaults.chipColors(backgroundColor = MaterialTheme.colors.primaryVariant, contentColor = Color.White)
                if (vMareeTable.isPortSelected(watch.portId)){
                    vColorPort =ChipDefaults.chipColors(backgroundColor = MaterialTheme.colors.secondaryVariant, contentColor = Color.White)
                }

                WatchAppChip(
                    watchModelNumber = watch.portIndex,
                    watchName = watch.portName,
                    onClickWatch = onClickWatch,

                    colors= vColorPort //ChipDefaults.chipColors(backgroundColor = MaterialTheme.colors.primaryVariant, contentColor = Color.White)
                )
            }

            coroutineScope.launch {
                // Animate scroll to the first item
                columnState.state.animateScrollToItem(vIndexScroll)
            }
        }
    }

}



@Composable
fun WatchListScreen1(
    columnState: ScalingLazyColumnState,
    showVignette: Boolean,
    onClickVignetteToggle: (Boolean) -> Unit,
    onClickWatch: (Int) -> Unit,
    vMareeTable : MareeTable
) {


    WatchListScreen(
        watches = WatchLocalDataSource().portsCoteOuest,
        columnState = columnState,
        showVignette = showVignette,
        onClickVignetteToggle = onClickVignetteToggle,
        onClickWatch = onClickWatch,
        modifier = Modifier,
        vMareeTable
    )

}


@Composable
fun WatchListScreenDt(
    watchesDt: List<WatchModelDt>,
    columnState: ScalingLazyColumnState,
    showVignette: Boolean,
    onClickVignetteToggle: (Boolean) -> Unit,
    onClickWatch: (Int) -> Unit,
    modifier: Modifier = Modifier,
    vIndexScroll: Int = 0

) {

    val state = rememberScrollState()
    val listState = rememberLazyListState()
    val coroutineScope = rememberCoroutineScope()

    //LaunchedEffect(Unit) { state.animateScrollTo(100) }
    ScreenScaffold(scrollState = columnState) {
        /*
         * The Horologist [ScalingLazyColumn] takes care of the horizontal and vertical
         * padding for the list, so there is no need to specify it, as in the [GreetingScreen]
         * composable.
         */
        ScalingLazyColumn(
            modifier = modifier.fillMaxSize(),
            columnState = columnState,
        ) {
            item {
                Text(text = "Dates Marées")
            }
            // Displays all watches.
            //items(watches,key={it.modelId}) { watch ->
            items(watchesDt) { watch ->
                WatchAppCards(
                    watchModelNumber = watch.modelId,
                    watchDateName = formatDateMaree(watch.dtString),
                    tabMaree = watch.tabMaree,
                    onClickWatch = onClickWatch,
                )
            }

            coroutineScope.launch {
                // Animate scroll to the first item
                columnState.state.animateScrollToItem(vIndexScroll)
            }
        }
    }
}
@Composable
fun WatchListScreenDt(
    columnState: ScalingLazyColumnState,
    showVignette: Boolean,
    onClickVignetteToggle: (Boolean) -> Unit,
    onClickWatch: (Int) -> Unit,
    vMareeTable : MareeTable
) {

    val vIndexScroll = 0
    WatchListScreenDt(
        watchesDt = vMareeTable.vDatesMaree,
        columnState = columnState,
        showVignette = showVignette,
        onClickVignetteToggle = onClickVignetteToggle,
        onClickWatch = onClickWatch,
        modifier = Modifier,
        vIndexScroll
    )

}





